/*:
 * @target MZ
 * @plugindesc ステート時に与ダメージの○％が自分に返ってくる反動ダメージ v1.4（反動時アニメーション＋ログ表示付き） 
 * @author ChatGPT
 *
 * @param RecoilAnimationId
 * @text 反動アニメーションID
 * @type animation
 * @default 0
 * @desc 反動ダメージを受けたときに再生するアニメーションID（0で無効）
 *
 * @param ShowRecoilPopup
 * @text 反動ダメージのポップアップ表示
 * @type boolean
 * @on 表示する
 * @off 表示しない
 * @default true
 * @desc 反動ダメージのポップアップ（頭上の数字）を表示するかどうか
 *
 * @help
 * 【概要】
 * ・特定のステートにかかっている間だけ、
 *   「与えたダメージの○％が自分に反動ダメージとして返ってくる」効果をつけます。
 * ・反動ダメージを受けたキャラに、指定したアニメーションを再生できます。
 * ・反動ダメージ用のダメージポップアップの表示／非表示を切り替えできます。
 *
 * 【ステートのメモ欄】
 *   <RecoilRate:30>
 *   → このステートにかかっている状態で与えたHPダメージの30％が
 *      自分に反動ダメージとして返ってきます。
 *
 * 【仕様】
 * ・アクター/敵キャラどちらにも有効。
 * ・全体攻撃のときは「各対象への与ダメージ合計 × ％」で反動ダメージを計算。
 * ・反動ダメージはHPを直接減らすため、防御・属性・耐性などの軽減は一切入りません。
 * ・複数の反動ステートを同時に持っている場合、％は合計されます。
 *   例：<RecoilRate:30> と <RecoilRate:20> → 合計50％。
 * ・反動ダメージが発生したときだけ、プラグインパラメータで指定した
 *   アニメーションIDをそのキャラに再生します（0ならアニメーションなし）。
 * ・反動ダメージは、バトルログに
 *   「○○は反動で △△ のダメージを受けた！」と表示されます。
 *
 * 【注意】
 * ・HPダメージを与えなかった場合（ミス・0ダメージなど）は反動も発生しません。
 * ・このプラグインではスキルや装備のメモ欄は見ていません。
 *   「ステートでのみ反動を管理」するシンプル仕様です。
 */

(() => {
    'use strict';

    const pluginName = 'RecoilDamageState';
    const parameters = PluginManager.parameters(pluginName);
    const recoilAnimationId = Number(parameters['RecoilAnimationId'] || 0);
    const showRecoilPopup = parameters['ShowRecoilPopup'] === 'true';

    // -------------------------------------------------------
    // 反動ダメージ用の一時変数を初期化
    // -------------------------------------------------------

    const _BattleManager_startAction = BattleManager.startAction;
    BattleManager.startAction = function() {
        const subject = this._subject;
        if (subject) {
            subject._recoilDamage = 0;
        }
        _BattleManager_startAction.call(this);
    };

    // -------------------------------------------------------
    // 与ダメージを監視して反動用に蓄積
    // -------------------------------------------------------

    const _Game_Action_apply = Game_Action.prototype.apply;
    Game_Action.prototype.apply = function(target) {
        const subject = this.subject();
        const prevHp = target.hp; // 事前のHP

        _Game_Action_apply.call(this, target);

        // HPへ影響する攻撃だったかどうか
        if (!subject) return;
        if (!this.isHpEffect()) return;

        // 実際に減ったHP（0以下なら何もしない）
        const damage = Math.max(0, prevHp - target.hp);
        if (damage <= 0) return;

        // この時点の反動率（ステートから取得）
        const rate = this.recoilRateByStates(subject);
        if (rate === 0) return;

        // 蓄積（全体攻撃のために合計しておく）
        subject._recoilDamage = (subject._recoilDamage || 0) + Math.floor(damage * rate);
    };

    // -------------------------------------------------------
    // ステートから反動率を取得
    // 例：<RecoilRate:30> → 0.30
    // -------------------------------------------------------

    Game_Action.prototype.recoilRateByStates = function(subject) {
        if (!subject || !subject.states) return 0;
        let rate = 0;

        const states = subject.states();
        for (const state of states) {
            if (state && state.meta && state.meta.RecoilRate) {
                rate += Number(state.meta.RecoilRate) / 100;
            }
        }

        return rate;
    };

    // -------------------------------------------------------
    // アクション終了時に蓄積された反動ダメージをまとめて適用
    // -------------------------------------------------------

    const _BattleManager_endAction = BattleManager.endAction;
BattleManager.endAction = function() {
    const subject = this._subject;
    if (subject && subject._recoilDamage && subject._recoilDamage > 0) {
        const value = subject._recoilDamage;
        subject._recoilDamage = 0;

        if (subject.isAlive()) {
            const result = subject.result();
            result.clear();
            result.hpDamage = value;

            // ★バトルログに反動ダメージ表示
            const logWindow = this._logWindow;
            if (logWindow) {
                const text = `${subject.name()}は反動で ${value} のダメージを受けた！`;
                logWindow.push('addText', text);
                logWindow.push('wait');
                logWindow.push('waitForEffect');
            }

            // HPを直接減らす（軽減なし）
            subject.gainHp(-value);

            // ダメージポップアップ
            subject.startDamagePopup();

            // 反動アニメーション（IDが設定されていれば）
            if (recoilAnimationId > 0) {
                $gameTemp.requestAnimation([subject], recoilAnimationId, false);
            }

            if (subject.isDead()) {
                subject.performCollapse();
            }
        }
    }

    _BattleManager_endAction.call(this);
};

})();
